package httpServer;

import java.util.*;

/**
 * Diese Klasse stellt einen Request dar 
 *
 * @author  Thorsten Fischer
 * @version 1.0, 30.05.2000
 */
public class Request {
  
  /** Die Major-Versionsnummer des HTTP-Protokolls (RFC 1945 3.1) */
  private int httpVerMajor; 
  /** Die Minor-Versionsnummer des HTTP-Protokolls (RFC 1945 3.1) */
  private int httpVerMinor; 
  /** falls true, mssen noch weitere Zeilen des Heads empfangen werden */
  private boolean isRecvHead = true;
  /** falls true, mu noch ein Entity empfangen werden */
  private boolean isRecvEntity = false;
  /** Die requestLine */
  private String requestLine=null;
  /** 
   * Hashtabelle, die die Header eines Request abspeichert. (RFC 1945 5 und 4.2)
   * Die field-names einer Headerzeile sind die keys des Hashs <BR> 
   * die filed-values einer Headerzeile sind die values des Hashs <BR>
   * (beides vom Typ String); die field-names sind Kleinbuchstaben
   * (case-insensitive nach RFC 1945 4.2)
   */
  private Hashtable header = new Hashtable();
  /** Die Request-Method (RFC 1945 5.1.1) */
  private String method = null;
  /** Der Request-URI aus der Request-Line (RFC 1945 5.1) */
  private String uri = null;
  
  
  /** 
   * Erzeugt ein neues Request-Objekt, das erst noch empfangen werden
   * mu. Deshalb wird <code>isRecvHead</code> auf true gesetzt.
   *
   * @see Request#isRecvHead
   */
  public Request() { isRecvHead = true; }
  
  /**
   * Parst die Request-Line (RFC 1945 5.1)
   * <p>
   * Wenn es sich um einen Request nach http/0.9 handelt,
   * werden weder Header noch Entitys empfangen und die Flags 
   * <code>isRecvHead</code> und <code>isRecvEntity</code>
   * false.
   *
   * @param rl Die Request-Line, wie sie vom UA empfangen wurde
   * @exception RequestException wenn die Request-Line ungltig war.
   *            Die <code>RequestException</code> enthlt dann den
   *            staus-code nach RFC 1945 6.1.1
   */
  public synchronized void setRequestLine(String rl) throws RequestException {
    requestLine = rl;
    //////////////////////////////////////////////////////////////////
    // Prfe zunchst, ob eine bekannte Methode vorliegt. (RFC 1945 5.1.1)
    for (int i=0; i<Http_1_0.method.size(); i++) {
      String aMethod = (String) Http_1_0.method.elementAt(i);
      method = rl.startsWith(aMethod)?aMethod:method;
    }
    if (method == null) throw new RequestException("501"); // Methode nicht untersttzt
    rl = rl.substring(method.length());                    // der Rest
    //////////////////////////////////////////////////////////////////
    // ermittele die HTTP-Version (RFC 1945 3.1 und 5)
    int posVer = rl.indexOf("HTTP/"); // Position der Version in rl
    if (posVer == -1) {
      // keine Versionsangabe, dann darf 
      // innerhalb des Request-URI kein weiteres SP vorkommen
      if (rl.trim().indexOf(Http_1_0.SP)>-1) throw new RequestException("400"); // Bad Request
      // also HTTP/0.9
      httpVerMajor=0;
      httpVerMinor=9;
      isRecvEntity = false;
    } else {
      String ver = rl.substring(posVer+5); // Der String nach "HTTP/"
      int posPunkt = ver.indexOf(".");     // Position des "." in ver
      if (posPunkt == -1) throw new RequestException("400"); // Bad request
      try {
        httpVerMajor=Integer.parseInt(ver.substring(0,posPunkt));
        httpVerMinor=Integer.parseInt(ver.substring(posPunkt+1));
      } catch (NumberFormatException e) {
        throw new RequestException("400"); // Bad request
      }
      rl = rl.substring(0,posVer); // Der Rest
    }
    //////////////////////////////////////////////////////////////////
    // Der briggebliebene String ist der Requst-URI
    uri = rl.trim();
    /*
    System.out.println("Der geparste Request:");
    System.out.println( method );
    System.out.println( httpVerMajor );
    System.out.println( httpVerMinor );
    System.out.println( uri );
    */
  } // setRequestLine
  
  /**
   * Gibt die original vom Client empfangene Request-Line
   * zurck (z.B. fr Logging)
   */
  public String getRequestLine() { return requestLine; }
  
  /**
   * Fgt eine empfangene Headerzeile dem Request hinzu.
   * <P>
   * Wenn <code>http_1_0.CRLF</code> bergeben wurde, wird das Flag
   * <code>isRecvHead</code> auf false gesetzt.
   * <P>
   * Ansonsten wird der bergebene String wird am Doppelpunkt 
   * getrennt. (RFC 1945 4.2) Befindet sich kein Doppelpunkt in der 
   * Headerzeile, wird die Zeile komplett ignoriert.
   * <P>
   * Der field-name wird in Kleinbuchstaben konvertiert
   * (case-insensitive) (RFC 1945 4.2)
   *
   * @param h Die Headerzeile, so, wie sie vom UA empfangen wurde 
   * @see Request#isRecvHead
   * @see Http_1_0#CRLF
   */
  public synchronized void addHeaderLine( String h ) {
    if (h==null) return;
    if (h.length()==0) {
      isRecvHead = false; // Es wurde ein CRLF empfangen, der Header ist fertig eingelesen
    } else if (h.indexOf(":")>=0) {
      String fieldName = h.substring(0,h.indexOf(":")).trim().toLowerCase();
      String fieldValue = h.substring(h.indexOf(":")+1).trim();
      /*
      System.out.println("field-name="+fieldName);
      System.out.println("field-value="+fieldValue);
      */
      header.put(fieldName,fieldValue); 
    } 
  } // addHeaderLine
  
  /** Gibt die Major-Zahl der HTTP-Version des Requests zurck */
  public int getHTTPVerMajor() { return httpVerMajor; }

  /** Gibt die Minor-Zahl der HTTP-Version des Requests zurck */
  public int getHTTPVerMinor() { return httpVerMinor; }

  /** 
   * Prft, ob die HTTP-Versionsnummer des Requests mit der
   * bergebenen bereinstimmt.
   *
   * @param major Die Major-Versionsnummer (RFC 1945 3.1)
   * @param minor Die Minor-Versionsnummer (RFC 1945 3.1)
   */
  public boolean isHTTPVer(int major, int minor) {
    return ((major==httpVerMajor) && (minor==httpVerMinor));
  } // isHttpVer
  
  /** Gibt zurck, ob noch weitere header empfangen werden mssen */
  public boolean isReceivingHead() { return isRecvHead; }

  /**
   * Gibt zurck, ob ein Entity empfangen werden mu.
   * Dies ist nie der Fall bei HTTP/0.9 und sonst
   * nur, wenn ein RequestHeader "Content-Length"
   * empfangen wurde. (RFC 1945 7.2)
   */
  public boolean isReceivingEntity() {
    return isHTTPVer(0,9)?false:header.containsKey("content-length");
  } // isReceivingEntity
  
  /**
   * Prft, ob der bergebene Header (filed-name) im Request
   * vorkommt.
   *
   * @param fieldName der field-Name des Headers, der gesucht wird.
   * @return den field-vaule des Headers <code>fieldName</code>, wenn
   *         er im Header des Requests vorkommt, <code>null</code> sonst.
   */
  public String getHeaderValue( String fieldName ) {
    return (String)header.get(fieldName.toLowerCase());
  } // getHeaderValue
  
  /**
   * Liefert den uri des Requests, oder <code>null</code> wenn keines
   * angesprochen war.
   */
  public String getUri() { return uri; }
  
  /** Liefert die Methode des Requests zurck */
  public String getMethod() { return method; }
  
} // Request