package dateisystem;

import java.io.*;
import java.util.*;

import socketVerbindung.*;
import httpServer.*;

/**
 * Diese Klasse stellt eine angeforderten Ressource dar 
 *
 * @author  Thorsten Fischer
 * @version 1.0, 30.05.2000
 */
public class Ressource {
  
  /** 
   * Die Content-Types, angesprochen ber die Extension des Files.
   * Als Keys in dem Hash dienen die Extensions der files. (String) <BR>
   * Als Values im Hash sind die entsprechenden Content-Types abgelegt
   * (String)<BR>
   * Beispiel: "htm", "text/html"
   */
  public static Hashtable types = new Hashtable();
  
  /* 
   * Initialisiert die Konstanten
   */
  static {
    types.put( "htm" , "text/html" );
    types.put( "html", "text/html" );
    types.put( "gif",  "image/gif" );
    types.put( "jpg",  "image/jpg" );
    // usw.
  }
  
  /** Die Datei, die die Ressource beinhaltet */
  private File res = null;
  /** Das Datum der letzten nderung der Datei */
  private Date lastModified = null;
  
  /** Keine Erzeugung dieser Objekte ohne weitere Angaben */
  private Ressource() {}
  
  /** 
   * Erzeugt ein Ressource-Objekt
   * 
   * @param name Der Pfad und Name zur Datei
   * @param basis Der Basispfad, der vor <code>name</code> angehngt wird
   * @exception IOException Wenn die Datei nicht existiert 
   *                        oder kein Zugriff erlaubt ist.
   *                        Die Exception hat als Fehlertext den 
   *                        entsprechenden HTTP-Statuscode
   */
  public Ressource( String basis, String name ) throws IOException {
    res = new File( basis + File.separator + name );
    if (!res.exists()) throw new IOException("404");
    if (!res.canRead()) throw new IOException("403");
    if (res.isDirectory()) throw new IOException("403");
    lastModified = new Date (res.lastModified());
  } // Ressource
  
  /** Gibt das Datum der letzten nderung der Ressource zurck */
  public Date getLastModified() { return lastModified; }
  
  /** Gibt den absoluten URI fr die Ressource zurck */
  public String getLocation() {
    String pfad = "";
    try {
      pfad = res.getCanonicalPath(); // Versuche alle relativen Angaben aufzulsen 
    } catch (IOException e) {
      pfad = res.getAbsolutePath();  // Dann halt nicht!
    }
    if (pfad.startsWith(Server.rootDir)) {
      // Ziehe den Basispfad wieder ab.
      pfad = pfad.substring(Server.rootDir.length());
    }
    pfad = pfad.replace(File.separatorChar, '/'); // Windows?
    return Server.URL+pfad;
  } // getLocation
  
  /** Gibt die Lnge der Ressource in Bytes zurck. */
  public long length() { return res.length(); }
  
  /** 
   * Gibt den Content-Type des Files zurck
   * abhngig von der Extension des Files
   *
   * @see Ressource#types
   */
  public String getType() {
    String name = res.getName();
    int posPunkt = name.lastIndexOf(".");
    if ( posPunkt > -1) {
      return (String) types.get(name.substring(posPunkt+1));
    } else {
      return null;
    }
  } // getType
  
  /**
   * Diese Methode schreibt die Ressource in den DataOutputStream
   *
   * @param out Der <code>DataOutputStream</code>, in den die Ressource
   *            geschrieben wird
   */
  public void write( DataOutputStream out ) throws SendException {
    try {
      BufferedInputStream in = new BufferedInputStream (new FileInputStream( res ),16384);
      BufferedOutputStream bufOut = new BufferedOutputStream(out,16384);
      long len=res.length();
      for (int i=0; i<len; i++) {
        bufOut.write(in.read());
      }
      bufOut.flush();
      in.close();
    } catch (IOException e) {
      e.printStackTrace();
      throw new SendException("Die Ressource konnte nicht an den Client gesendet werden");
    }
  } // write
  
} // class Ressource